# Employee Management System - Setup Guide

## Current Issue

You're getting this error:
```
{"status":"error","message":"Database error: SQLSTATE[42S22]: Column not found: 1054 Unknown column 'e.role_id' in 'on clause'"}
```

This happens because the `tbl_employees` table is missing the new columns we added.

## Solution: Run the Migration

### Step 1: Run the Database Migration

1. **Open your browser** and go to:
   ```
   http://localhost/steve/migrate.php
   ```

2. **The page will automatically:**
   - Check for missing columns (role_id, designation_id, user_id, password)
   - Add any missing columns
   - Create indexes for better performance
   - Show you a success message

3. **You should see:**
   ```
   ✅ MIGRATION COMPLETED SUCCESSFULLY!
   
   Table structure updated:
   - role_id column: ✓
   - designation_id column: ✓
   - user_id column: ✓
   - password column: ✓
   ```

### Step 2: Verify the Migration

After migration, refresh your browser and check:

1. **Go to employee-list.php**
   ```
   http://localhost/steve/employee-list.php
   ```
   - Should show employee list without errors
   - Dropdowns should work

2. **Add a new employee**
   ```
   http://localhost/steve/employee-add.php
   ```
   - Branch dropdown should load ✓
   - Role dropdown should load ✓
   - Designation dropdown should load ✓
   - Username field should appear ✓
   - Password fields should appear ✓

3. **Test creating an employee:**
   - Fill in all required fields
   - Submit the form
   - Should see success message
   - New employee appears in list

## What the Migration Does

### Columns Added to `tbl_employees` table:

1. **role_id** (INT)
   - Links to the `roles` table
   - Position: After `branch_id`
   - Used for role selection dropdown

2. **designation_id** (INT)
   - Links to the `tbl_designations` table
   - Position: After `role_id`
   - Used for designation selection dropdown

3. **user_id** (VARCHAR)
   - For login username
   - Position: After `status`
   - Used for authentication

4. **password** (VARCHAR)
   - For login password (plain text, not encrypted)
   - Position: After `user_id`
   - Used for authentication

### Indexes Created:

1. **idx_role_id** - For faster role lookups
2. **idx_designation_id** - For faster designation lookups

## Files Involved

### Frontend
- `employee-add.php` - Form to add/edit employees
- `employee-list.php` - List all employees

### Backend APIs
- `api/employee/create.php` - Create new employee
- `api/employee/update.php` - Update employee
- `api/employee/read.php` - List employees
- `api/employee/read_single.php` - Get single employee
- `api/branch/read.php` - Load branches
- `api/role/read.php` - Load roles (NEW)
- `api/designation/read.php` - Load designations

### Migration
- `migrate.php` - Automatic migration script (NEW)
- `database/migrations/add_role_and_designation_to_employees.sql` - SQL migration file

## Employee Form Structure

### Fields Collected:

**Basic Info:**
- Branch (required, dropdown)
- Role (required, dropdown)
- Designation (required, dropdown)
- Name (required, text)

**Contact Info:**
- Email (optional)
- Phone (optional)
- Address (optional)
- City, State, Country, Pincode (optional)

**Personal Info:**
- Age (optional)
- Father Name (optional)
- Mother Name (optional)
- Education (optional)

**Employment Info:**
- Salary (optional)
- Experience (optional)
- Status (required, dropdown)

**Authentication:**
- Username (required, text)
- Password (required, password field)
- Confirm Password (required, must match)

## Database Schema

### Updated `tbl_employees` Table:

```sql
CREATE TABLE `tbl_employees` (
  `id` int(11) AUTO_INCREMENT PRIMARY KEY,
  `branch_id` int(11) NOT NULL,
  `role_id` int(11) DEFAULT NULL,
  `designation_id` int(11) DEFAULT NULL,
  `name` varchar(255) NOT NULL,
  `age` int(11) DEFAULT NULL,
  `email` varchar(255) DEFAULT NULL,
  `father_name` varchar(255) DEFAULT NULL,
  `mother_name` varchar(255) DEFAULT NULL,
  `education` varchar(255) DEFAULT NULL,
  `salary` decimal(10,2) DEFAULT NULL,
  `experience` varchar(100) DEFAULT NULL,
  `phone` varchar(20) NOT NULL,
  `address` text DEFAULT NULL,
  `city` varchar(100) DEFAULT NULL,
  `pincode` varchar(10) DEFAULT NULL,
  `state` varchar(100) DEFAULT NULL,
  `country` varchar(100) DEFAULT 'India',
  `status` enum('active','inactive') NOT NULL DEFAULT 'active',
  `user_id` varchar(100) DEFAULT NULL,
  `password` varchar(255) DEFAULT NULL,
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
  `updated_by` int(11) DEFAULT NULL,
  `updated_at` timestamp NULL DEFAULT NULL,
  
  KEY `idx_branch_id` (`branch_id`),
  KEY `idx_role_id` (`role_id`),
  KEY `idx_designation_id` (`designation_id`),
  KEY `idx_status` (`status`),
  KEY `idx_user_id` (`user_id`)
);
```

## Testing Checklist

After running the migration, test these:

### ✓ Employee List
- [ ] Page loads without errors
- [ ] Employee list shows (if any employees exist)
- [ ] Branch filter loads
- [ ] Designation filter loads
- [ ] Status filter works

### ✓ Add New Employee
- [ ] All dropdowns load (Branch, Role, Designation)
- [ ] Username field is visible
- [ ] Password fields are visible
- [ ] Form validates required fields
- [ ] Password confirmation works
- [ ] Can submit form
- [ ] New employee appears in list

### ✓ Edit Employee
- [ ] Employee data loads in form
- [ ] Branch dropdown shows correct value
- [ ] Role dropdown shows correct value
- [ ] Designation dropdown shows correct value
- [ ] Username pre-fills
- [ ] Password pre-fills
- [ ] Can update and submit

### ✓ Validation
- [ ] Cannot submit without branch
- [ ] Cannot submit without role
- [ ] Cannot submit without designation
- [ ] Cannot submit without name
- [ ] Cannot submit without username
- [ ] Cannot submit without password
- [ ] Password mismatch error shows
- [ ] Phone validation (if entered must be 10 digits)
- [ ] Email validation (if entered must be valid)

## Troubleshooting

### Issue: Still getting "Column not found" error

**Solution:**
1. Clear browser cache (Ctrl+Shift+Delete)
2. Run migrate.php again
3. Check if migration completed successfully
4. Refresh employee-list.php

### Issue: Dropdowns not loading

**Solution:**
1. Check browser console (F12) for errors
2. Verify API endpoints are returning data
3. Check database has data (branches, roles, designations)

### Issue: Cannot add employee

**Solution:**
1. Fill all required fields
2. Ensure passwords match
3. Check browser console for errors
4. Verify API endpoint is responding

### Issue: "Database error" messages

**Solution:**
1. Check database connection in config/db.php
2. Verify table structure using migrate.php
3. Check MySQL error logs

## Next Steps

1. ✅ **Run migrate.php** - http://localhost/steve/migrate.php
2. ✅ **Test employee list** - http://localhost/steve/employee-list.php
3. ✅ **Add new employee** - http://localhost/steve/employee-add.php
4. ✅ **Edit employee** - Click edit on any employee
5. ✅ **Verify authentication fields** - Check username/password are saved

## Files to Keep

Keep these files for reference:
- `migrate.php` - For future schema updates
- `EMPLOYEE_FORM_COMPLETE_FIX.md` - Complete form documentation
- `USERNAME_PASSWORD_FIELDS_UPDATE.md` - Auth fields documentation
- `EMPLOYEE_DROPDOWN_FIX.md` - Dropdown fix documentation
- `database/migrations/add_role_and_designation_to_employees.sql` - SQL migration

## Important Notes

⚠️ **Passwords are stored in PLAIN TEXT** (not encrypted)
- This is as per your requirement
- For production, consider using password hashing

✅ **All dropdowns use `length=-1`** to fetch all records
- Ensures complete list of branches, roles, designations

✅ **Form validates locally** before sending to server
- Better user experience
- Reduces unnecessary API calls

✅ **Both create and update** support all new fields
- Consistent across form

## Support

If you encounter any issues:

1. Check **SETUP_GUIDE.md** (this file)
2. Check **EMPLOYEE_FORM_COMPLETE_FIX.md** for form details
3. Check **USERNAME_PASSWORD_FIELDS_UPDATE.md** for auth details
4. Run `migrate.php` to verify database schema
5. Check browser console (F12) for errors

## Quick Links

- **Employee List:** http://localhost/steve/employee-list.php
- **Add Employee:** http://localhost/steve/employee-add.php
- **Run Migration:** http://localhost/steve/migrate.php
- **PhpMyAdmin:** http://localhost/phpmyadmin (to view database)
