# Tickets Module Implementation

## Overview
A comprehensive Ticket Management system for tracking and managing customer support tickets across branches, clients, and employees.

## Files Created

### 1. **tickets.php** - Main Listing Page
- **Location:** `D:\xampp\htdocs\steve\tickets.php`
- **Features:**
  - DataTables integration for responsive table display
  - Multi-filter system:
    - Branch Filter
    - Client Filter
    - Employee Filter (Assigned To)
    - Status Filter (Open, In Progress, Resolved, Closed)
    - Priority Filter (High, Medium, Low)
    - Date Range Filter
  - Color-coded status badges and priority indicators
  - View, Edit, Delete actions per ticket
  - Export functionality (CSV, Excel, PDF, Print)
  - Permission-based access (view, add, edit, delete)

### 2. **api/ticket/read.php** - DataTables API Endpoint
- **Location:** `D:\xampp\htdocs\steve\api\ticket\read.php`
- **Features:**
  - Server-side pagination and filtering
  - Joins with tbl_branch, tbl_client, tbl_employee tables
  - Supports all filter parameters
  - Date range filtering
  - Search functionality across ticket fields
  - Returns data in DataTables format

### 3. **api/ticket/delete.php** - Delete API Endpoint
- **Location:** `D:\xampp\htdocs\steve\api\ticket\delete.php`
- **Features:**
  - Secure ticket deletion with permission check
  - Returns JSON response with status

## Database Requirements

### Table: `tbl_tickets`
```sql
CREATE TABLE tbl_tickets (
    id INT PRIMARY KEY AUTO_INCREMENT,
    ticket_number VARCHAR(50) UNIQUE NOT NULL,
    title VARCHAR(255) NOT NULL,
    description LONGTEXT,
    priority ENUM('High', 'Medium', 'Low') DEFAULT 'Medium',
    status ENUM('Open', 'In Progress', 'Resolved', 'Closed') DEFAULT 'Open',
    branch_id INT,
    client_id INT,
    employee_id INT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (branch_id) REFERENCES tbl_branch(id),
    FOREIGN KEY (client_id) REFERENCES tbl_client(id),
    FOREIGN KEY (employee_id) REFERENCES tbl_employee(id)
);
```

## UI Features

### Filters (Responsive Layout)
- **Branch Filter:** Dropdown populated from `api/branch/read.php`
- **Client Filter:** Dropdown populated from `api/client/read.php`
- **Employee Filter:** Dropdown populated from `api/employee/get_employees.php`
- **Status Filter:** Fixed options (Open, In Progress, Resolved, Closed)
- **Priority Filter:** Fixed options (High, Medium, Low)
- **Date Range Picker:** With preset ranges (Today, Yesterday, Last 7/30 Days, This/Last Month)

### Table Columns
1. ID
2. Ticket # (Bold text)
3. Title (Clickable link to ticket-view.php)
4. Branch
5. Client
6. Assigned To (Employee Name)
7. Priority (Color-coded: Red for High, Orange for Medium, Green for Low)
8. Status (Badges: Blue for Open, Yellow for In Progress, Green for Resolved, Gray for Closed)
9. Created (Date)
10. Actions (View, Edit, Delete with icon buttons)

### Styling
- **Status Badges:**
  - Open: Light blue background (#e7f3ff)
  - In Progress: Light yellow background (#fff3cd)
  - Resolved: Light green background (#d4edda)
  - Closed: Light gray background (#f8f9fa)

- **Priority Text:**
  - High: Red (#dc3545)
  - Medium: Orange (#fd7e14)
  - Low: Green (#28a745)

## JavaScript Functionality

### Data Loading
```javascript
- Load branches from api/branch/read.php
- Load clients from api/client/read.php
- Load employees from api/employee/get_employees.php
- Initialize Select2 for dropdown styling
- Initialize DateRangePicker for date filtering
```

### DataTables Configuration
- **Processing:** true (Shows processing message)
- **ServerSide:** true (Server-side processing)
- **AJAX:** Calls api/ticket/read.php with filters
- **Pagination:** Dynamic based on records
- **Sorting:** By created_at DESC (newest first)
- **Export Buttons:** CSV, Excel, PDF, Print

### Event Handlers
- **Filter Button Click:** Triggers table reload
- **Reset Button Click:** Clears all filters and reloads
- **Filter Change:** Auto-reload table on any filter change
- **Delete Button Click:** Confirms deletion and calls api/ticket/delete.php

## Integration Points

### Required API Endpoints (Already Exist)
- `/api/branch/read.php` - Get branches
- `/api/client/read.php` - Get clients
- `/api/employee/get_employees.php` - Get employees

### Required Pages (To be Created)
- `ticket-add.php` - Create new ticket
- `ticket-edit.php?id=X` - Edit ticket
- `ticket-view.php?id=X` - View ticket details

### Required Permissions (Middleware)
- `ticket:is_view` - View tickets
- `ticket:is_add` - Create tickets
- `ticket:is_edit` - Edit tickets
- `ticket:is_delete` - Delete tickets

## Features Summary

✅ **List all tickets** with pagination
✅ **Filter by multiple criteria** (branch, client, employee, status, priority, date)
✅ **Search** across title, ticket number, description
✅ **Color-coded status indicators**
✅ **Priority level highlighting**
✅ **Date range filtering** with preset options
✅ **Export to CSV/Excel/PDF/Print**
✅ **View ticket details** link
✅ **Edit ticket** action
✅ **Delete ticket** with confirmation
✅ **Permission-based access control**
✅ **Responsive design** (mobile-friendly)
✅ **Select2 dropdown styling**
✅ **Real-time filter** application

## Next Steps

1. Create the database table `tbl_tickets`
2. Create `ticket-add.php` for creating new tickets
3. Create `ticket-edit.php` for editing tickets
4. Create `ticket-view.php` for viewing ticket details
5. Add ticket-related API endpoints (create, update, view single)
6. Add menu item to sidebar linking to `tickets.php`
7. Configure permissions in your permission management system

## Notes

- All table joins are LEFT JOINs to handle NULL values gracefully
- The module uses existing authentication and middleware
- Select2 is configured with a minimum search threshold of 5 results
- Date formatting is handled by JavaScript (toLocaleDateString)
- Delete operations require confirmation to prevent accidental deletion
