# Company List - Style Guide

This document outlines the styling standards and design patterns used in the Company Management interface.

## Button Styles

### Size Standard
All buttons use the **small size** variant for consistency:
- Class: `btn-sm`
- Applied to: Primary actions, filters, table actions

### Button Types

#### Primary Actions
- **Class**: `btn btn-sm btn-soft-primary`
- **Usage**: Main actions (Add, Edit, Search)
- **Examples**:
  - New Company button
  - Search button
  - Edit button in table actions

#### Light Actions
- **Class**: `btn btn-sm btn-light`
- **Usage**: Reset/clear actions, export functions
- **Examples**:
  - Reset filter button
  - Export dropdown button

#### View Actions
- **Class**: `btn btn-sm btn-outline-dark`
- **Usage**: Read-only view actions
- **Examples**:
  - View button in table actions

#### Destructive Actions
- **Class**: `btn btn-sm btn-soft-danger`
- **Usage**: Delete/remove actions
- **Examples**:
  - Delete button in table actions

## Form Controls

### Input Fields
- **Class**: `form-control form-control-sm`
- **Font Size**: 13px
- **Padding**: 0.25rem 0.5rem

### Select Dropdowns
- **Class**: `form-select form-select-sm`
- **Font Size**: 13px
- **Padding**: 0.25rem 0.5rem

### Date Range Picker
- **Class**: `btn btn-sm btn-white border`
- **Display**: Flex with icon, text, and chevron
- **Font Size**: 12px (label text)

## Status Display

### Active Status
- **Style**: Text only (no background)
- **Class**: `text-success fw-semibold`
- **Color**: Green text
- **Weight**: Semi-bold

### Inactive Status
- **Style**: Text only (no background)
- **Class**: `text-danger fw-semibold`
- **Color**: Red text
- **Weight**: Semi-bold

## Icons

### Icon Library
- **Library**: Tabler Icons (`ti` classes)

### Common Icons
- `ti-plus` - Add/Create actions
- `ti-search` - Search functionality
- `ti-rotate` - Reset/Refresh actions
- `ti-download` - Export/Download actions
- `ti-eye` - View details
- `ti-edit` - Edit functionality
- `ti-trash` - Delete actions
- `ti-calendar` - Date selection
- `ti-chevron-down` - Dropdown indicators

### Icon Spacing
- Add margin class `me-1` after icon when followed by text

## Table Styling

### Table Classes
- **Base**: `table table-hover dt-responsive nowrap w-100`
- **Features**: Hover effect, responsive, no text wrapping

### Table Font
- **Color**: Black (#000000) for all text
- **Size**: 13px
- **Applied via**: Custom CSS override

### Action Column
- **Width**: 150px
- **Layout**: Horizontal flex with gap
- **Gap**: `gap-1` (0.25rem)
- **Buttons**: Small size with icons
- **Edit Redirect**: Edit button links to `company-add.php?id=` (shared add/edit form)

## Layout Components

### Page Title Section
- **Layout**: Flexbox (row on desktop, column on mobile)
- **Padding**: `py-3` (vertical padding)
- **Alignment**: Space between title and action button

### Filter Section
- **Margin**: `mb-3` (bottom margin)
- **Layout**: Responsive grid (col-md-*)
- **Components**: Date range (col-md-3), Text inputs (col-md-2), Buttons (col-md-3)

### Card Container
- **Classes**: `card` > `card-body`
- **Usage**: Wraps filters and table

## Color Overrides

### Custom CSS
```css
/* Force black text in table */
#companyTable,
#companyTable * {
    color: #000000 !important;
}

/* Remove default text colors */
.text-primary, .text-info, .text-warning,
.text-success, .text-danger {
    color: #000 !important;
}

/* Small form controls */
.form-control-sm, .form-select-sm {
    padding: 0.25rem 0.5rem !important;
    font-size: 13px !important;
}
```

## DataTable Configuration

### Export Buttons
- **Style**: `btn btn-sm btn-light dropdown-toggle`
- **Icon**: `ti-download`
- **Options**: Copy, CSV, Excel, Print, PDF
- **Background**: Light background for subtle appearance

### Pagination
- **Icons**: Tabler chevron icons
- **Page Length**: 25 rows per page

## Responsive Behavior

### Mobile Layout
- Page title section switches to column layout
- Filter inputs stack vertically
- Table uses responsive plugin for horizontal scrolling

## Best Practices

1. **Always use `btn-sm`** for buttons to maintain size consistency
2. **Use soft variants** (`btn-soft-*`) for primary and danger actions
3. **Use outline variant** (`btn-outline-*`) for neutral view actions
4. **Use light variant** (`btn-light`) for utility actions (reset, export)
5. **Font size 13px** for all form controls and table text
6. **No background colors** for status indicators (text colors only)
7. **Icons always precede text** in buttons with `me-1` spacing
8. **Gap-1 spacing** between action buttons in tables
9. **Edit functionality** uses the same form as add (company-add.php?id=)

## Permission-Based Display

### Conditional Rendering
- Add button: Requires `can_add('company')`
- Edit button: Requires `canEdit` permission
- Delete button: Requires `canDelete` permission

### Implementation
```javascript
const userPermissions = {
    canEdit: <?php echo $can_edit; ?>,
    canDelete: <?php echo $can_delete; ?>
};
```

---

## Recent Updates

### 2026-02-16 (Latest)
- Changed reset and export buttons from `btn-secondary` to `btn-light` for lighter appearance
- Edit button now redirects to `company-add.php?id=` instead of `company-edit.php?id=`
- Added `ti-download` icon to common icons list
- Updated best practices to include light variant usage

**File**: company-list.php
