<?php

// Prevent multiple inclusions
if (!defined('HELPER_INCLUDED')) {
    define('HELPER_INCLUDED', true);

    // Include database connection if not already loaded
    if (!isset($pdo)) {
        require_once __DIR__ . '/config.php';
    }

    // Start session if not already started
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }

    $role_id = $_SESSION['role_id'] ?? 1; // Default role_id is 1 (Admin)

    function get_permission($permission, $can)
    {
        global $role_id; // Ensure access to global $role_id

        if ($role_id === 1) {
            return true; // Admin has all permissions
        }

        $permissions = get_staff_permissions();

        foreach ($permissions as $permObject) {
            if ($permObject['permission_prefix'] === $permission) {
                return !empty($can) && isset($permObject[$can]) && $permObject[$can] == '1';
            }
        }

        return false;
    }

    function get_staff_permissions()
    {
        global $pdo, $role_id; // Access global variables

        // Check if database connection exists
        if (!$pdo) {
            error_log("Database connection not available in get_staff_permissions");
            return []; // Return empty array if no connection
        }

        try {
            $sql = "SELECT sp.*, p.id AS permission_id, p.prefix AS permission_prefix 
                    FROM staff_privileges sp
                    JOIN permission p ON p.id = sp.permission_id 
                    WHERE sp.role_id = ?";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([$role_id]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: []; // Return empty array if no permissions found
        } catch (PDOException $e) {
            error_log("Database error in get_staff_permissions: " . $e->getMessage());
            return []; // Return empty array on error
        }
    }

    function access_denied()
    {
        global $site_url;

        $redirect_url = 'index.php';
        echo "<script>alert('Access Denied'); window.location.href='$redirect_url';</script>";
        exit();
    }

    function ajax_access_denied()
    {
        echo json_encode(['status' => 'access_denied']);
        exit();
    }

    /**
     * Get current user's information including user_type
     */
    function get_current_user_info()
    {
        global $pdo;

        if (!isset($_SESSION['username'])) {
            return null;
        }

        try {
            $stmt = $pdo->prepare("SELECT id, username, user_id, role_id, user_type, status 
                                   FROM tbl_user 
                                   WHERE username = ?");
            $stmt->execute([$_SESSION['username']]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Error fetching user info: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Check if current user is an employee
     */
    function is_employee()
    {
        // Check session first for better performance
        if (isset($_SESSION['user_type'])) {
            return $_SESSION['user_type'] === 'employee';
        }

        // Fallback to database query
        $userInfo = get_current_user_info();
        return $userInfo && $userInfo['user_type'] === 'employee';
    }

    /**
     * Check if current user is a regular user (not employee)
     */
    function is_user_only()
    {
        // Check session first for better performance
        if (isset($_SESSION['user_type'])) {
            return $_SESSION['user_type'] === 'user';
        }

        // Fallback to database query
        $userInfo = get_current_user_info();
        return $userInfo && $userInfo['user_type'] === 'user';
    }

    /**
     * Check if current user is both employee and user
     */
    function is_both()
    {
        // Check session first for better performance
        if (isset($_SESSION['user_type'])) {
            return $_SESSION['user_type'] === 'both';
        }

        // Fallback to database query
        $userInfo = get_current_user_info();
        return $userInfo && $userInfo['user_type'] === 'both';
    }

    /**
     * Get current user's employee ID from tbl_employee
     */
    function get_current_user_employee_id()
    {
        global $pdo;

        // Check for both regular and mobile sessions
        if (!isset($_SESSION['username']) && !isset($_SESSION['mobile_username'])) {
            return null;
        }

        // For mobile sessions, return the employee ID directly
        if (isset($_SESSION['mobile_employee_id'])) {
            return $_SESSION['mobile_employee_id'];
        }

        try {
            // First get user info from tbl_user
            $userInfo = get_current_user_info();
            if (!$userInfo) {
                return null;
            }

            // If user_type is 'employee' or 'both', find the corresponding employee record
            if (in_array($userInfo['user_type'], ['employee', 'both'])) {
                // Try to match by tbl_user.user_id = tbl_employee.id
                $stmt = $pdo->prepare("
                    SELECT e.id
                    FROM tbl_employee e
                    WHERE e.id = ? AND e.status = 'active'
                ");
                $stmt->execute([$userInfo['user_id']]);
                $employee = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($employee) {
                    return $employee['id'];
                }

                // If no match found, try to match by email
                $stmt = $pdo->prepare("
                    SELECT e.id
                    FROM tbl_employee e
                    WHERE e.email = ? AND e.status = 'active'
                ");
                $stmt->execute([$userInfo['username']]);
                $employee = $stmt->fetch(PDO::FETCH_ASSOC);

                if ($employee) {
                    return $employee['id'];
                }
            }

            return null;
        } catch (PDOException $e) {
            error_log("Error getting employee ID: " . $e->getMessage());
            return null;
        }
    }
    /**
     * Fetch extended user details (Employee Name, Designation)
     * Returns an array with 'name' and 'designation'
     */
    function fetch_user_extended_details($user_id, $username, $role_id, $user_type)
    {
        global $pdo;

        $fullName = $username; // Fallback
        $designation = 'User'; // Fallback

        // Get Role Name as default designation
        try {
            $stmtRole = $pdo->prepare("SELECT name FROM roles WHERE id = :role_id");
            $stmtRole->execute([':role_id' => $role_id]);
            $roleRow = $stmtRole->fetch(PDO::FETCH_ASSOC);
            if ($roleRow) {
                $designation = $roleRow['name'];
            }
        } catch (Exception $e) {
            // Ignore role fetch error
            error_log("Error fetching role: " . $e->getMessage());
        }

        // If employee, get employee specific details
        if (in_array($user_type, ['employee', 'both'])) {
            try {
                $stmtEmp = $pdo->prepare("SELECT e.name as employee_name, d.name as designation_name
                                          FROM tbl_employee e
                                          LEFT JOIN tbl_designation d ON e.designation_id = d.id
                                          WHERE e.id = :uid OR e.email = :uname");
                $stmtEmp->execute([':uid' => $user_id, ':uname' => $username]);
                $empInfo = $stmtEmp->fetch(PDO::FETCH_ASSOC);

                if ($empInfo) {
                    if (!empty($empInfo['employee_name'])) {
                        $fullName = $empInfo['employee_name'];
                    }
                    if (!empty($empInfo['designation_name'])) {
                        $designation = $empInfo['designation_name'];
                    }
                }
            } catch (Exception $e) {
                // Ignore employee fetch error
                error_log("Error fetching employee details: " . $e->getMessage());
            }
        }

        return ['name' => $fullName, 'designation' => $designation];
    }

    /**
     * Get the display name of the current logged-in user
     */
    function get_logged_user_name()
    {
        if (isset($_SESSION['employee_name'])) {
            return $_SESSION['employee_name'];
        }
        return $_SESSION['username'] ?? 'User';
    }

    /**
     * Get the designation/role of the current logged-in user
     */
    function get_logged_user_designation()
    {
        return $_SESSION['designation'] ?? 'Role';
    }

    /**
     * Handle Image Upload with Compression (under 50KB)
     * @param array $file_input - The $_FILES element
     * @param string $folder - Subfolder in assets/images/
     * @param string|null $old_file - Path of the old file to delete
     * @param int $max_kb - Maximum file size in KB
     * @return string|false - The new relative file path or false on failure
     */
    function handle_image_upload($file_input, $folder, $old_file = null, $max_kb = 50)
    {
        if (!isset($file_input) || $file_input['error'] !== UPLOAD_ERR_OK) {
            error_log("Upload error: " . ($file_input['error'] ?? 'No file input'));
            return false;
        }

        $base_dir = __DIR__ . "/../assets/images/" . trim($folder, '/') . "/";
        if (!file_exists($base_dir)) {
            if (!mkdir($base_dir, 0777, true)) {
                error_log("Failed to create directory: $base_dir");
                return false;
            }
        }

        $extension = strtolower(pathinfo($file_input['name'], PATHINFO_EXTENSION));
        $new_filename = uniqid() . '.' . $extension;
        $target_path = $base_dir . $new_filename;
        $relative_path = "assets/images/" . trim($folder, '/') . "/" . $new_filename;

        // Compression logic using GD
        $tmp_name = $file_input['tmp_name'];
        $image_info = getimagesize($tmp_name);
        if (!$image_info) {
            error_log("Failed to get image size for $tmp_name");
            return false;
        }
        list($width, $height, $type) = $image_info;

        $image = null;
        switch ($type) {
            case IMAGETYPE_JPEG:
                if (function_exists('imagecreatefromjpeg')) {
                    $image = imagecreatefromjpeg($tmp_name);
                }
                break;
            case IMAGETYPE_PNG:
                if (function_exists('imagecreatefrompng')) {
                    $image = imagecreatefrompng($tmp_name);
                    // Convert extension to jpg for output
                    $target_path = preg_replace('/\.(png)$/i', '.jpg', $target_path);
                    $relative_path = preg_replace('/\.(png)$/i', '.jpg', $relative_path);
                }
                break;
            case IMAGETYPE_GIF:
                if (function_exists('imagecreatefromgif')) {
                    $image = imagecreatefromgif($tmp_name);
                }
                break;
        }

        if (!$image) {
            error_log("Failed to create image resource or GD functions missing for type $type");
            return false;
        }

        // Start with quality 80 and reduce until under max_kb
        $quality = 80;
        $success = false;
        do {
            ob_start();
            if (imagejpeg($image, null, $quality)) {
                $content = ob_get_clean();
                $size = strlen($content);
                if ($size <= $max_kb * 1024 || $quality <= 10) {
                    if (file_put_contents($target_path, $content) !== false) {
                        $success = true;
                    } else {
                        error_log("Failed to write file to $target_path");
                    }
                    break;
                }
            } else {
                ob_end_clean();
                error_log("imagejpeg failed at quality $quality");
                break;
            }
            $quality -= 10;
        } while ($quality > 0);

        imagedestroy($image);

        if (!$success) {
            error_log("Compression/Save failed for $target_path");
            return false;
        }

        // Delete old file if provided
        if ($old_file) {
            delete_image($old_file);
        }

        return $relative_path;
    }

    /**
     * Delete an image file from the system
     * @param string $file_path - Relative path (e.g., assets/images/...)
     * @return bool
     */
    /**
     * Delete an image file from the system
     * @param string $file_path - Relative path (e.g., assets/images/...)
     * @return bool
     */
    function delete_image($file_path)
    {
        if (!$file_path)
            return false;
        $full_path = __DIR__ . "/../" . $file_path;
        if (file_exists($full_path) && is_file($full_path)) {
            return unlink($full_path);
        }
        return false;
    }

    /**
     * Sanitize text input
     * @param string $input
     * @return string
     */
    function sanitizeText($input)
    {
        return htmlspecialchars(strip_tags(trim($input)));
    }
}

?>